<?php
/******************** DEBUG ********************/
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

/******************** INCLUDES ********************/
require_once __DIR__ . '/../../app/core/helpers.php';
require_once __DIR__ . '/../../app/core/Auth.php';
require_once __DIR__ . '/../../app/config/database.php';

Auth::requireLogin();

$title  = "Nouvelle simulation";
$active = 'simulations';

$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

/* ======================================================
   PRÉ-SÉLECTION LOT (si venant depuis la liste des lots)
   ====================================================== */
$preselected_lot_id     = (int)($_GET['lot_id'] ?? 0);
$preselected_project_id = 0;

if ($preselected_lot_id > 0) {
    $stmt = $pdo->prepare("
        SELECT project_id 
        FROM lots 
        WHERE id = ?
    ");
    $stmt->execute([$preselected_lot_id]);
    $preselected_project_id = (int)$stmt->fetchColumn();
}

/******************** LOTS ********************/
$lots = $pdo->query("
    SELECT 
        l.id,
        l.project_id,
        l.reference,
        l.price,
        lt.name AS lot_type
    FROM lots l
    JOIN lot_types lt ON lt.id = l.lot_type_id
    WHERE l.status = 'disponible'
    ORDER BY lt.name, l.reference
")->fetchAll(PDO::FETCH_ASSOC);

/******************** CLIENTS ********************/
$clients = $pdo->query("
    SELECT id, fullname, phone
    FROM clients
    ORDER BY fullname
")->fetchAll(PDO::FETCH_ASSOC);

/******************** PROJETS ********************/
$projects = $pdo->query("
    SELECT id, name
    FROM projects
    ORDER BY name
")->fetchAll(PDO::FETCH_ASSOC);

$errors = [];

/******************** POST ********************/
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    $client_id  = (int)($_POST['client_id'] ?? 0);
    $project_id = (int)($_POST['project_id'] ?? 0);
    $lot_id     = (int)($_POST['lot_id'] ?? 0);

    $amount_to_pay = (float)($_POST['amount_to_pay'] ?? 0);
    $paid_percent  = (float)($_POST['paid_percent'] ?? 0);

    if ($client_id <= 0)  $errors[] = "Client invalide";
    if ($project_id <= 0) $errors[] = "Projet invalide";
    if ($lot_id <= 0)     $errors[] = "Lot invalide";

    if ($amount_to_pay <= 0 && $paid_percent <= 0) {
        $errors[] = "Veuillez saisir un montant OU un pourcentage à payer";
    }

    $stmt = $pdo->prepare("SELECT price FROM lots WHERE id = ? AND project_id = ?");
    $stmt->execute([$lot_id, $project_id]);
    $lot = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$lot) {
        $errors[] = "Lot introuvable";
    }

    if (!$errors) {

        $total_price = (float)$lot['price'];

        if ($amount_to_pay > 0) {
            $paid_percent = ($amount_to_pay / $total_price) * 100;
        } else {
            $amount_to_pay = ($paid_percent / 100) * $total_price;
        }

        // Remise progressive 0 → 6 %
        if ($paid_percent >= 100) {
            $discount_percent = 6;
        } elseif ($paid_percent > 60) {
            $discount_percent = (($paid_percent - 60) / 40) * 6;
        } else {
            $discount_percent = 0;
        }

        $final_price = $total_price * (1 - ($discount_percent / 100));

        $stmt = $pdo->prepare("
            INSERT INTO simulations
            (client_id, lot_id, total_price, paid_percent, discount_percent, final_price, status)
            VALUES (?, ?, ?, ?, ?, ?, 'en_attente')
        ");

        $stmt->execute([
            $client_id,
            $lot_id,
            $total_price,
            $paid_percent,
            $discount_percent,
            $final_price
        ]);

        redirect('/public/simulations/index.php');
        exit;
    }
}

include __DIR__ . '/../../app/views/layout/header.php';
include __DIR__ . '/../../app/views/layout/sidebar.php';
?>

<script>
const ALL_LOTS = <?= json_encode($lots) ?>;
</script>

<section class="content pt-3">
<div class="container-fluid">

<div class="card">
<div class="card-header">
    <h3 class="card-title">Nouvelle simulation</h3>
</div>

<div class="card-body">

<?php if ($errors): ?>
<div class="alert alert-danger">
<ul class="mb-0">
<?php foreach ($errors as $e): ?>
<li><?= e($e) ?></li>
<?php endforeach; ?>
</ul>
</div>
<?php endif; ?>

<form method="post">

<!-- champs cachés si lot pré-sélectionné -->
<?php if ($preselected_lot_id): ?>
<input type="hidden" name="lot_id" value="<?= $preselected_lot_id ?>">
<input type="hidden" name="project_id" value="<?= $preselected_project_id ?>">
<?php endif; ?>

<div class="row">

<div class="col-md-4">
<label>Client *</label>
<select name="client_id" class="form-control" required>
<option value="">-- Choisir --</option>
<?php foreach ($clients as $c): ?>
<option value="<?= $c['id'] ?>">
<?= e($c['fullname']) ?> (<?= e($c['phone']) ?>)
</option>
<?php endforeach; ?>
</select>
</div>

<div class="col-md-4">
<label>Projet *</label>
<select id="project_id" name="project_id" class="form-control"
<?= $preselected_project_id ? 'disabled' : '' ?> required>
<option value="">-- Choisir --</option>
<?php foreach ($projects as $p): ?>
<option value="<?= $p['id'] ?>"
<?= $p['id'] == $preselected_project_id ? 'selected' : '' ?>>
<?= e($p['name']) ?>
</option>
<?php endforeach; ?>
</select>
</div>

<div class="col-md-4">
<label>Lot *</label>
<select name="lot_id" id="lot_id" class="form-control"
<?= $preselected_lot_id ? 'disabled' : '' ?> required>
<option value="">-- Sélectionner --</option>
<?php foreach ($lots as $l): ?>
<?php if ($l['id'] == $preselected_lot_id): ?>
<option value="<?= $l['id'] ?>" selected data-price="<?= $l['price'] ?>">
<?= e($l['lot_type'].' - '.$l['reference']) ?>
</option>
<?php endif; ?>
<?php endforeach; ?>
</select>
</div>

</div>

<hr>

<div class="row">
<div class="col-md-4">
<label>Montant total</label>
<input type="text" id="total_price" class="form-control" readonly>
</div>

<div class="col-md-4">
<label>Montant à payer</label>
<input type="text" name="amount_to_pay" id="amount_to_pay" class="form-control">
</div>

<div class="col-md-4">
<label>% à payer</label>
<input type="number" step="0.01" name="paid_percent" id="paid_percent" class="form-control">
</div>
</div>

<div class="row mt-3">
<div class="col-md-4">
<label>Remise (%)</label>
<input type="text" id="discount_percent" class="form-control" readonly>
</div>

<div class="col-md-4">
<label>Montant de la remise (DA)</label>
<input type="text" id="discount_amount" class="form-control" readonly>
</div>

<div class="col-md-4">
<label>Prix final à payer</label>
<input type="text" id="final_price" class="form-control" readonly>
</div>
</div>

<hr>

<button class="btn btn-success">
<i class="fas fa-save"></i> Enregistrer
</button>

<a href="/public/simulations/index.php" class="btn btn-secondary ml-2">
Annuler
</a>

</form>

</div>
</div>

</div>
</section>

<script>
function formatDA(value) {
    if (isNaN(value)) return '';
    return Number(value).toLocaleString('fr-FR', {
        minimumFractionDigits: 2,
        maximumFractionDigits: 2
    });
}

document.addEventListener('DOMContentLoaded', function () {

    const projectSelect  = document.getElementById('project_id');
    const lotSelect      = document.getElementById('lot_id');

    const totalInput     = document.getElementById('total_price');
    const amountInput    = document.getElementById('amount_to_pay');
    const percentInput   = document.getElementById('paid_percent');
    const discountInput  = document.getElementById('discount_percent');
    const discountAmtInp = document.getElementById('discount_amount');
    const finalInput     = document.getElementById('final_price');

    let lotPrice = 0;

    function resetCalc(resetTotal = false) {
        if (resetTotal) {
            lotPrice = 0;
            totalInput.value = '';
        }
        amountInput.value = '';
        percentInput.value = '';
        discountInput.value = '';
        discountAmtInp.value = '';
        finalInput.value = '';
    }

    if (lotSelect.options.length === 1 && lotSelect.options[0].dataset.price) {
        lotPrice = parseFloat(lotSelect.options[0].dataset.price);
        totalInput.value = formatDA(lotPrice);
    }

    projectSelect?.addEventListener('change', function () {

        const projectId = parseInt(this.value);
        lotSelect.innerHTML = '';
        lotSelect.disabled = true;
        resetCalc(true);

        if (!projectId) return;

        const filteredLots = ALL_LOTS.filter(lot => lot.project_id === projectId);

        if (!filteredLots.length) return;

        lotSelect.innerHTML = '<option value="">-- Choisir un lot --</option>';

        filteredLots.forEach(lot => {
            const opt = document.createElement('option');
            opt.value = lot.id;
            opt.dataset.price = lot.price;
            opt.textContent = `${lot.lot_type} - ${lot.reference}`;
            lotSelect.appendChild(opt);
        });

        lotSelect.disabled = false;
    });

    lotSelect.addEventListener('change', function () {
        const opt = this.options[this.selectedIndex];
        lotPrice = parseFloat(opt.dataset.price || 0);
        resetCalc();
        if (lotPrice > 0) totalInput.value = formatDA(lotPrice);
    });

    function calculate(from) {
        if (!lotPrice) return;

        let amount  = parseFloat(amountInput.value.replace(/\s/g, ''));
        let percent = parseFloat(percentInput.value);

        if (from === 'amount' && amount > 0) {
            percent = (amount / lotPrice) * 100;
            percentInput.value = percent.toFixed(2);
        }

        if (from === 'percent' && percent > 0) {
            amount = (percent / 100) * lotPrice;
            amountInput.value = formatDA(amount);
        }

        let discountPercent = 0;
        if (percent >= 100) discountPercent = 6;
        else if (percent > 60) discountPercent = ((percent - 60) / 40) * 6;

        const discountAmount = lotPrice * (discountPercent / 100);
        const finalPrice = lotPrice - discountAmount;

        discountInput.value  = discountPercent.toFixed(2);
        discountAmtInp.value = formatDA(discountAmount);
        finalInput.value     = formatDA(finalPrice);
    }

    amountInput.addEventListener('input', () => {
        percentInput.value = '';
        calculate('amount');
    });

    percentInput.addEventListener('input', () => {
        amountInput.value = '';
        calculate('percent');
    });

});
</script>

<?php include __DIR__ . '/../../app/views/layout/footer.php'; ?>
