<?php
require_once __DIR__ . '/../../app/core/helpers.php';
require_once __DIR__ . '/../../app/core/Auth.php';
require_once __DIR__ . '/../../app/config/database.php';
require_once __DIR__ . '/../../app/core/Permissions.php';
require_once __DIR__ . '/../../app/core/Audit.php';

Auth::requireLogin();
Permissions::allow(['admin','director','manager','commercial']);

$simulation_id = (int)($_GET['simulation_id'] ?? 0);
if ($simulation_id <= 0) redirect('/public/simulations/index.php');

$simStmt = $pdo->prepare("
  SELECT s.*, l.id AS lot_id, l.reference, l.status AS lot_status
  FROM simulations s
  JOIN lots l ON l.id = s.lot_id
  WHERE s.id = ?
");
$simStmt->execute([$simulation_id]);
$sim = $simStmt->fetch();

if (!$sim) redirect('/public/simulations/index.php');

// Optionnel : si tu utilises un statut "acceptee", décommente la ligne suivante
// if (($sim['status'] ?? '') !== 'acceptee') { die("Simulation non acceptée."); }

if ($sim['lot_status'] !== 'disponible') {
    die("Lot non disponible (déjà réservé/vendu/bloqué).");
}

$errors = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    // 7 jours
    $expires = date('Y-m-d H:i:s', strtotime('+7 days'));

    $pdo->beginTransaction();
    try {
        // Créer réservation
        $stmt = $pdo->prepare("
          INSERT INTO reservations
          (lot_id, simulation_id, client_id, client_name, client_phone, reserved_at, expires_at, status)
          VALUES (?, ?, ?, ?, ?, NOW(), ?, 'active')
        ");
        $stmt->execute([
            (int)$sim['lot_id'],
            (int)$simulation_id,
            $sim['client_id'] ?: null,
            $sim['client_name'],
            $sim['client_phone'],
            $expires
        ]);

        $reservation_id = (int)$pdo->lastInsertId();

        // Lot -> réservé
        $u = $pdo->prepare("UPDATE lots SET status='reserve' WHERE id=?");
        $u->execute([(int)$sim['lot_id']]);

        // Audit
        $commercial_id = null;
        if ($sim['client_id']) {
            $c = $pdo->prepare("SELECT owner_user_id FROM clients WHERE id=?");
            $c->execute([(int)$sim['client_id']]);
            $commercial_id = (int)($c->fetchColumn() ?: 0);
        }

        Audit::log([
          'client_id' => $sim['client_id'] ?: null,
          'commercial_id' => $commercial_id ?: null,
          'action_type' => 'create_reservation',
          'entity_type' => 'reservation',
          'entity_id' => $reservation_id,
          'description' => 'Réservation créée (expire sous 7 jours)'
        ]);

        $pdo->commit();

        redirect('/public/reservations/index.php');

    } catch (Exception $e) {
        $pdo->rollBack();
        $errors[] = "Erreur réservation : ".$e->getMessage();
    }
}

$title  = "Créer réservation";
$active = 'reservations';

include __DIR__ . '/../../app/views/layout/header.php';
include __DIR__ . '/../../app/views/layout/sidebar.php';
?>

<section class="content pt-3">
<div class="container-fluid">

<div class="card">
  <div class="card-header"><h3 class="card-title">Créer une réservation</h3></div>
  <div class="card-body">

    <?php if ($errors): ?>
      <div class="alert alert-danger">
        <ul class="mb-0"><?php foreach($errors as $e): ?><li><?= e($e) ?></li><?php endforeach; ?></ul>
      </div>
    <?php endif; ?>

    <p><strong>Lot :</strong> <?= e($sim['reference']) ?></p>
    <p><strong>Client :</strong> <?= e($sim['client_name']) ?> <?= $sim['client_phone'] ? ' | '.$sim['client_phone'] : '' ?></p>
    <p><strong>Prix final :</strong> <?= number_format((float)$sim['final_price'],0,' ',' ') ?> DA</p>
    <p><strong>Expiration :</strong> 7 jours</p>

    <form method="post">
      <button class="btn btn-info">
        <i class="fas fa-bookmark"></i> Confirmer la réservation
      </button>
      <a class="btn btn-secondary ml-2" href="<?= e(base_url('/public/simulations/index.php')) ?>">Annuler</a>
    </form>

  </div>
</div>

</div>
</section>

<?php include __DIR__ . '/../../app/views/layout/footer.php'; ?>
