<?php
require_once __DIR__ . '/../../app/core/helpers.php';
require_once __DIR__ . '/../../app/core/Auth.php';
require_once __DIR__ . '/../../app/config/database.php';

Auth::requireLogin();

$reservation_id = (int)($_GET['id'] ?? 0);
if ($reservation_id <= 0) {
    redirect('/public/reservations/index.php');
}

try {
    $pdo->beginTransaction();

    // 1) Charger réservation + lot + prix final depuis simulation
    $stmt = $pdo->prepare("
        SELECT 
            r.id AS reservation_id,
            r.status AS reservation_status,
            r.lot_id,
            l.status AS lot_status,
            s.final_price
        FROM reservations r
        JOIN lots l ON l.id = r.lot_id
        JOIN simulations s ON s.id = r.simulation_id
        WHERE r.id = ?
        FOR UPDATE
    ");
    $stmt->execute([$reservation_id]);
    $row = $stmt->fetch();

    if (!$row) {
        throw new Exception("Réservation introuvable.");
    }

    // Interdire si déjà vendue / confirmée
    if ($row['lot_status'] === 'vendu' || $row['reservation_status'] === 'confirmee') {
        throw new Exception("Cette vente est déjà confirmée.");
    }

    // 2) Calculer total payé
    $stmt = $pdo->prepare("
        SELECT COALESCE(SUM(amount),0)
        FROM payments
        WHERE reservation_id = ?
    ");
    $stmt->execute([$reservation_id]);
    $total_paid = (float)$stmt->fetchColumn();

    $final_price = (float)$row['final_price'];

    if ($final_price <= 0) {
        throw new Exception("Prix final invalide. Vérifie la simulation.");
    }

    // 3) Condition de vente : >= 60%
    $ratio = $total_paid / $final_price;

    if ($ratio < 0.60) {
        throw new Exception("Paiement insuffisant : minimum 60% requis pour confirmer la vente.");
    }

    // 4) Mettre le lot en VENDU
    $stmt = $pdo->prepare("UPDATE lots SET status='vendu' WHERE id=?");
    $stmt->execute([$row['lot_id']]);

    // 5) Confirmer la réservation
    $stmt = $pdo->prepare("
        UPDATE reservations
        SET status='confirmee', sold_at = NOW()
        WHERE id=?
    ");
    $stmt->execute([$reservation_id]);

    $pdo->commit();

    redirect('/public/reservations/index.php?msg=vente_confirmee');

} catch (Exception $e) {
    $pdo->rollBack();
    // Simple redirection avec message (tu peux l'afficher plus tard)
    redirect('/public/reservations/index.php?err=' . urlencode($e->getMessage()));
}
