<?php
ini_set('display_errors', 1);
error_reporting(E_ALL);

require_once __DIR__ . '/../../app/core/helpers.php';
require_once __DIR__ . '/../../app/core/Auth.php';
require_once __DIR__ . '/../../app/config/database.php';
require_once __DIR__ . '/../../app/core/Permissions.php';

Auth::requireLogin();
Permissions::allow(['admin','director','manager','finance']);

$title = "Nouvel achat | IMMOTECH";
$active = 'purchases';
$subactive = 'purchases_create';

function post($key, $default = null) {
  return $_POST[$key] ?? $default;
}

$errors = [];

// Charger fournisseurs / projets
$suppliers = $pdo->query("SELECT id, name FROM suppliers ORDER BY name ASC")->fetchAll(PDO::FETCH_ASSOC);
$projects  = $pdo->query("SELECT id, name FROM projects ORDER BY name ASC")->fetchAll(PDO::FETCH_ASSOC);

// Valeurs par défaut form
$old_supplier_id  = (int)($_POST['supplier_id'] ?? 0);
$old_project_id   = $_POST['project_id'] ?? '';
$old_invoice_no   = $_POST['invoice_no'] ?? '';
$old_invoice_date = $_POST['invoice_date'] ?? '';
$old_status       = $_POST['status'] ?? 'DRAFT';
$old_note         = $_POST['note'] ?? '';

$old_desc  = $_POST['description'] ?? [''];
$old_qty   = $_POST['qty'] ?? ['1'];
$old_price = $_POST['unit_price'] ?? ['0'];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {

  $supplier_id  = (int) post('supplier_id', 0);
  $project_id   = (post('project_id', '') !== '') ? (int)post('project_id') : null;
  $invoice_no   = trim((string)post('invoice_no', ''));
  $invoice_date = post('invoice_date', '') ?: null;
  $status       = post('status', 'DRAFT');
  $note         = trim((string)post('note', ''));

  $descriptions = post('description', []);
  $qtys         = post('qty', []);
  $prices       = post('unit_price', []);

  if ($supplier_id <= 0) $errors[] = "Veuillez sélectionner un fournisseur.";
  if (!in_array($status, ['DRAFT','VALIDATED','PARTIAL_PAID','PAID'], true)) $status = 'DRAFT';

  // Valider lignes
  $lines = [];
  $total_ht = 0.0;

  if (!is_array($descriptions) || count($descriptions) === 0) {
    $errors[] = "Veuillez ajouter au moins une ligne d'achat.";
  } else {
    for ($i = 0; $i < count($descriptions); $i++) {
      $desc = trim((string)($descriptions[$i] ?? ''));
      $qty  = (float) str_replace(',', '.', (string)($qtys[$i] ?? 0));
      $unit = (float) str_replace(',', '.', (string)($prices[$i] ?? 0));

      if ($desc === '' && $qty == 0 && $unit == 0) continue;

      if ($desc === '') { $errors[] = "Ligne " . ($i+1) . ": description obligatoire."; continue; }
      if ($qty <= 0)     { $errors[] = "Ligne " . ($i+1) . ": quantité doit être > 0."; continue; }
      if ($unit < 0)     { $errors[] = "Ligne " . ($i+1) . ": prix unitaire invalide."; continue; }

      $line_total = round($qty * $unit, 2);
      $total_ht += $line_total;

      $lines[] = [
        'description' => $desc,
        'qty' => $qty,
        'unit_price' => $unit,
        'total' => $line_total,
      ];
    }

    if (count($lines) === 0) $errors[] = "Veuillez saisir au moins une ligne valide.";
  }

  $total_ht  = round($total_ht, 2);
  $total_ttc = $total_ht;

  if (empty($errors)) {
    try {
      $pdo->beginTransaction();

      // Insert purchase
      $sql = "INSERT INTO purchases
        (supplier_id, project_id, invoice_no, invoice_date, total_ht, total_ttc, status, note)
        VALUES
        (:supplier_id, :project_id, :invoice_no, :invoice_date, :total_ht, :total_ttc, :status, :note)";
      $stmt = $pdo->prepare($sql);
      $stmt->execute([
        ':supplier_id'  => $supplier_id,
        ':project_id'   => $project_id,
        ':invoice_no'   => $invoice_no !== '' ? $invoice_no : null,
        ':invoice_date' => $invoice_date,
        ':total_ht'     => $total_ht,
        ':total_ttc'    => $total_ttc,
        ':status'       => $status,
        ':note'         => $note !== '' ? $note : null,
      ]);

      $purchase_id = (int)$pdo->lastInsertId();

      // Insert lines
      $sqlLine = "INSERT INTO purchase_lines (purchase_id, description, qty, unit_price, total)
                  VALUES (:purchase_id, :description, :qty, :unit_price, :total)";
      $stmtLine = $pdo->prepare($sqlLine);

      foreach ($lines as $ln) {
        $stmtLine->execute([
          ':purchase_id' => $purchase_id,
          ':description' => $ln['description'],
          ':qty'         => $ln['qty'],
          ':unit_price'  => $ln['unit_price'],
          ':total'       => $ln['total'],
        ]);
      }

      $pdo->commit();
      header("Location: " . e(base_url('/public/purchases/index.php?created=1')));
      exit;

    } catch (Exception $e) {
      if ($pdo->inTransaction()) $pdo->rollBack();
      $errors[] = "Erreur lors de l'enregistrement : " . $e->getMessage();
    }
  }
}

include __DIR__ . '/../../app/views/layout/header.php';
include __DIR__ . '/../../app/views/layout/sidebar.php';
?>

<section class="content pt-3">
  <div class="container-fluid">

    <div class="d-flex justify-content-between align-items-center mb-3">
      <div>
        <h3 class="mb-0">Nouvel achat</h3>
        <small class="text-muted">Facture fournisseur + lignes d’achat</small>
      </div>
      <a href="<?= e(base_url('/public/purchases/index.php')) ?>" class="btn btn-secondary">
        <i class="fas fa-arrow-left mr-1"></i> Retour
      </a>
    </div>

    <?php if (!empty($errors)): ?>
      <div class="alert alert-danger">
        <ul class="mb-0 pl-3">
          <?php foreach ($errors as $err): ?>
            <li><?= e($err) ?></li>
          <?php endforeach; ?>
        </ul>
      </div>
    <?php endif; ?>

    <form method="post" id="purchaseForm">
      <div class="card">
        <div class="card-body">

          <div class="row">
            <div class="col-md-4">
              <label>Fournisseur *</label>
              <select class="form-control" name="supplier_id" required>
                <option value="">-- Sélectionner --</option>
                <?php foreach ($suppliers as $s): ?>
                  <option value="<?= (int)$s['id'] ?>" <?= $old_supplier_id === (int)$s['id'] ? 'selected' : '' ?>>
                    <?= e($s['name']) ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>

            <div class="col-md-3">
              <label>Projet (optionnel)</label>
              <select class="form-control" name="project_id">
                <option value="">-- Aucun --</option>
                <?php foreach ($projects as $p): ?>
                  <option value="<?= (int)$p['id'] ?>" <?= (string)$old_project_id === (string)$p['id'] ? 'selected' : '' ?>>
                    <?= e($p['name']) ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>

            <div class="col-md-2">
              <label>N° facture</label>
              <input class="form-control" type="text" name="invoice_no" value="<?= e($old_invoice_no) ?>" placeholder="FAC-2026-001">
            </div>

            <div class="col-md-2">
              <label>Date facture</label>
              <input class="form-control" type="date" name="invoice_date" value="<?= e($old_invoice_date) ?>">
            </div>

            <div class="col-md-1">
              <label>Statut</label>
              <select class="form-control" name="status">
               <?php foreach ([
              'DRAFT' => 'Brouillon',
              'VALIDATED' => 'Validé',
              'PARTIAL_PAID' => 'Partiellement payé',
              'PAID' => 'Payé'
            ] as $k => $v): ?>
              <option value="<?= $k ?>" <?= $old_status === $k ? 'selected' : '' ?>><?= $v ?></option>
            <?php endforeach; ?>

              </select>
            </div>
          </div>

          <div class="mt-3">
            <label>Note</label>
            <textarea class="form-control" name="note" rows="2" placeholder="Remarques..."><?= e($old_note) ?></textarea>
          </div>

          <hr class="my-4">

          <div class="d-flex justify-content-between align-items-center">
            <h5 class="mb-0">Lignes d’achat</h5>
            <button type="button" class="btn btn-outline-secondary btn-sm" id="btnAddLine">
              <i class="fas fa-plus mr-1"></i> Ajouter ligne
            </button>
          </div>

          <div class="table-responsive mt-2">
            <table class="table table-bordered" id="linesTable">
              <thead class="thead-light">
                <tr>
                  <th style="width:50%;">Description *</th>
                  <th style="width:15%;">Qté *</th>
                  <th style="width:15%;">PU</th>
                  <th style="width:15%;">Total</th>
                  <th style="width:5%;">Action</th>
                </tr>
              </thead>
              <tbody>
                <?php
                  $count = max(count($old_desc), 1);
                  for ($i=0; $i<$count; $i++):
                    $d = $old_desc[$i] ?? '';
                    $q = $old_qty[$i] ?? '1';
                    $u = $old_price[$i] ?? '0';
                ?>
                <tr>
                  <td><input class="form-control" type="text" name="description[]" value="<?= e($d) ?>" required></td>
                  <td><input class="form-control qty" type="number" step="0.01" min="0" name="qty[]" value="<?= e($q) ?>" required></td>
                  <td><input class="form-control unit" type="number" step="0.01" min="0" name="unit_price[]" value="<?= e($u) ?>"></td>
                  <td class="lineTotal text-right font-weight-bold">0.00</td>
                  <td class="text-center">
                    <button type="button" class="btn btn-danger btn-sm btnRemove"><i class="fas fa-times"></i></button>
                  </td>
                </tr>
                <?php endfor; ?>
              </tbody>
            </table>
          </div>

          <div class="d-flex justify-content-end mt-2">
            <div class="p-2 border rounded bg-light" style="min-width:240px;">
              <div class="d-flex justify-content-between">
                <span class="text-muted">Total HT</span>
                <strong id="totalHT">0.00</strong>
              </div>
              <div class="d-flex justify-content-between">
                <span class="text-muted">Total TTC</span>
                <strong id="totalTTC">0.00</strong>
              </div>
            </div>
          </div>

        </div>

        <div class="card-footer d-flex justify-content-end">
          <button type="submit" class="btn btn-success">
            <i class="fas fa-save mr-1"></i> Enregistrer
          </button>
        </div>
      </div>
    </form>

  </div>
</section>

<script>
function recalc() {
  let total = 0;
  document.querySelectorAll("#linesTable tbody tr").forEach(tr => {
    const qty = parseFloat((tr.querySelector(".qty")?.value || "0").replace(",", ".")) || 0;
    const unit = parseFloat((tr.querySelector(".unit")?.value || "0").replace(",", ".")) || 0;
    const lt = Math.round(qty * unit * 100) / 100;
    tr.querySelector(".lineTotal").textContent = lt.toFixed(2);
    total += lt;
  });
  total = Math.round(total * 100) / 100;
  document.getElementById("totalHT").textContent = total.toFixed(2);
  document.getElementById("totalTTC").textContent = total.toFixed(2);
}

document.getElementById("btnAddLine").addEventListener("click", () => {
  const tbody = document.querySelector("#linesTable tbody");
  const tr = document.createElement("tr");
  tr.innerHTML = `
    <td><input class="form-control" type="text" name="description[]" required></td>
    <td><input class="form-control qty" type="number" step="0.01" min="0" name="qty[]" value="1" required></td>
    <td><input class="form-control unit" type="number" step="0.01" min="0" name="unit_price[]" value="0"></td>
    <td class="lineTotal text-right font-weight-bold">0.00</td>
    <td class="text-center">
      <button type="button" class="btn btn-danger btn-sm btnRemove"><i class="fas fa-times"></i></button>
    </td>
  `;
  tbody.appendChild(tr);
  recalc();
});

document.addEventListener("click", (e) => {
  if (e.target.closest(".btnRemove")) {
    const tbody = document.querySelector("#linesTable tbody");
    if (tbody.querySelectorAll("tr").length <= 1) {
      tbody.querySelector("tr input[name='description[]']").value = '';
      tbody.querySelector("tr input[name='qty[]']").value = '1';
      tbody.querySelector("tr input[name='unit_price[]']").value = '0';
      recalc();
      return;
    }
    e.target.closest("tr").remove();
    recalc();
  }
});

document.addEventListener("input", (e) => {
  if (e.target.classList.contains("qty") || e.target.classList.contains("unit")) recalc();
});

recalc();
</script>

<?php include __DIR__ . '/../../app/views/layout/footer.php'; ?>
