<?php
ini_set('display_errors', 1);
error_reporting(E_ALL);

require_once __DIR__ . '/../../app/core/helpers.php';
require_once __DIR__ . '/../../app/core/Auth.php';
require_once __DIR__ . '/../../app/config/database.php';

Auth::requireLogin();

$title  = "Encaisser un paiement";
$active = 'payments';

$errors = [];

/* =========================
   MODE: OV OU LIBRE
========================= */
$ov_id = (int)($_GET['ov_id'] ?? 0);
$modeOv = $ov_id > 0;

$ov = null;
$lot_price = 0;
$total_paid_lot = 0;
$remaining_lot = 0;

/* =========================
   DATA LISTS (mode libre)
========================= */
$clients = $pdo->query("SELECT id, fullname FROM clients ORDER BY fullname")->fetchAll(PDO::FETCH_ASSOC);

$lots = $pdo->query("
  SELECT l.id, l.reference, l.price
  FROM lots l
  WHERE l.status IN ('reserve','vendu','disponible')
  ORDER BY l.reference
")->fetchAll(PDO::FETCH_ASSOC);

/* =========================
   CHARGER OV (si mode OV)
========================= */
if ($modeOv) {

    $stmt = $pdo->prepare("
        SELECT
            po.id,
            po.amount AS ov_amount,
            po.status AS ov_status,
            po.client_id,
            po.lot_id,
            c.fullname AS client_name,
            l.reference AS lot_reference,
            l.price AS lot_price
        FROM payment_orders po
        JOIN clients c ON c.id = po.client_id
        JOIN lots l ON l.id = po.lot_id
        WHERE po.id = ?
    ");
    $stmt->execute([$ov_id]);
    $ov = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$ov) {
        die("OV introuvable");
    }

    $lot_price = (float)$ov['lot_price'];

    // Total payé du lot (tous OVs confondus)
    $stmt = $pdo->prepare("
        SELECT IFNULL(SUM(p.amount),0)
        FROM payments p
        JOIN payment_orders po ON po.id = p.payment_order_id
        WHERE po.lot_id = ?
          AND p.status = 'valide'
    ");
    $stmt->execute([$ov['lot_id']]);
    $total_paid_lot = (float)$stmt->fetchColumn();

    $remaining_lot = max(0, $lot_price - $total_paid_lot);
}

/* =========================
   POST (encaisser)
========================= */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    $amount    = (float)($_POST['amount'] ?? 0);
    $method    = $_POST['method'] ?? '';
    $reference = trim($_POST['reference'] ?? '');

    // Champs mode libre
    $client_id = (int)($_POST['client_id'] ?? 0);
    $lot_id    = (int)($_POST['lot_id'] ?? 0);

    if ($amount <= 0) $errors[] = "Montant invalide";
    if (!$method)     $errors[] = "Mode de paiement requis";

    if (!$errors) {

        /* =========================
           SI MODE LIBRE: créer OV
        ========================= */
        if (!$modeOv) {

            if ($client_id <= 0) $errors[] = "Client invalide";
            if ($lot_id <= 0)    $errors[] = "Lot invalide";

            // Prix du lot
            $stmt = $pdo->prepare("SELECT price FROM lots WHERE id=?");
            $stmt->execute([$lot_id]);
            $lot_price = (float)$stmt->fetchColumn();

            if ($lot_price <= 0) $errors[] = "Lot introuvable / prix invalide";

            // Total payé lot
            $stmt = $pdo->prepare("
                SELECT IFNULL(SUM(p.amount),0)
                FROM payments p
                JOIN payment_orders po ON po.id = p.payment_order_id
                WHERE po.lot_id = ?
                  AND p.status = 'valide'
            ");
            $stmt->execute([$lot_id]);
            $total_paid_lot = (float)$stmt->fetchColumn();

            $remaining_lot = max(0, $lot_price - $total_paid_lot);

            if ($amount > $remaining_lot) {
                $errors[] = "Le montant dépasse le reste à payer du lot";
            }

            if (!$errors) {
                // Créer OV du montant saisi
                $stmt = $pdo->prepare("
                    INSERT INTO payment_orders (client_id, lot_id, amount, status, created_at)
                    VALUES (?, ?, ?, 'en_attente', NOW())
                ");
                $stmt->execute([$client_id, $lot_id, $amount]);
                $ov_id = (int)$pdo->lastInsertId();

                // Charger l’OV créé pour l’affichage/redirect
                $ov = [
                    'id' => $ov_id,
                    'ov_amount' => $amount,
                    'ov_status' => 'en_attente',
                    'client_id' => $client_id,
                    'lot_id' => $lot_id,
                ];
                $modeOv = true;
            }
        }

        /* =========================
           MODE OV: encaisser paiement
        ========================= */
        if ($modeOv && !$errors) {

            // sécuriser infos OV si on vient du mode OV initial
            if (!$ov) {
                // recharge minimale
                $stmt = $pdo->prepare("SELECT client_id, lot_id, amount AS ov_amount FROM payment_orders WHERE id=?");
                $stmt->execute([$ov_id]);
                $ov = $stmt->fetch(PDO::FETCH_ASSOC);
            }

            // Recalcul remaining lot
            $stmt = $pdo->prepare("SELECT price FROM lots WHERE id=?");
            $stmt->execute([(int)$ov['lot_id']]);
            $lot_price = (float)$stmt->fetchColumn();

            $stmt = $pdo->prepare("
                SELECT IFNULL(SUM(p.amount),0)
                FROM payments p
                JOIN payment_orders po ON po.id = p.payment_order_id
                WHERE po.lot_id = ?
                  AND p.status = 'valide'
            ");
            $stmt->execute([(int)$ov['lot_id']]);
            $total_paid_lot = (float)$stmt->fetchColumn();

            $remaining_lot = max(0, $lot_price - $total_paid_lot);

            if ($amount > $remaining_lot) {
                $errors[] = "Le montant dépasse le reste à payer du lot";
            }

            if (!$errors) {

                // Insert payment
                $stmt = $pdo->prepare("
                    INSERT INTO payments
                    (payment_order_id, amount, payment_date, method, reference, status)
                    VALUES (?, ?, NOW(), ?, ?, 'valide')
                ");
                $stmt->execute([$ov_id, $amount, $method, $reference ?: null]);

                // Client devient client
                $pdo->prepare("UPDATE clients SET status='client' WHERE id=?")->execute([(int)$ov['client_id']]);

                // Lot -> vendu (vente confirmée même si paiement partiel)
                $pdo->prepare("UPDATE lots SET status='vendu' WHERE id=?")->execute([(int)$ov['lot_id']]);

                // Reservation -> vendue (si existe)
                $pdo->prepare("UPDATE reservations SET status='vendue', sold_at=NOW() WHERE lot_id=?")->execute([(int)$ov['lot_id']]);

                // OV -> payee (1 OV = 1 paiement, donc on solde)
                $pdo->prepare("UPDATE payment_orders SET status='payee' WHERE id=?")->execute([$ov_id]);

                redirect("/public/payment_orders/show.php?id=".$ov_id);
                exit;
            }
        }
    }
}

include __DIR__ . '/../../app/views/layout/header.php';
include __DIR__ . '/../../app/views/layout/sidebar.php';
?>

<section class="content pt-3">
<div class="container-fluid">

<div class="card">
<div class="card-header">
  <h3 class="card-title"><?= $modeOv ? "Encaisser un paiement (OV #$ov_id)" : "Nouveau paiement" ?></h3>
</div>

<div class="card-body">

<?php if ($errors): ?>
  <div class="alert alert-danger">
    <ul class="mb-0">
      <?php foreach ($errors as $e): ?>
        <li><?= e($e) ?></li>
      <?php endforeach; ?>
    </ul>
  </div>
<?php endif; ?>

<form method="post">

<?php if (!$modeOv): ?>
  <div class="row">
    <div class="col-md-4">
      <label>Client *</label>
      <select name="client_id" class="form-control" required>
        <option value="">-- Choisir --</option>
        <?php foreach ($clients as $c): ?>
          <option value="<?= (int)$c['id'] ?>"><?= e($c['fullname']) ?></option>
        <?php endforeach; ?>
      </select>
    </div>

    <div class="col-md-4">
      <label>Lot *</label>
      <select name="lot_id" class="form-control" required>
        <option value="">-- Choisir --</option>
        <?php foreach ($lots as $l): ?>
          <option value="<?= (int)$l['id'] ?>">
            <?= e($l['reference']) ?> - <?= number_format((float)$l['price'], 0, ' ', ' ') ?> DA
          </option>
        <?php endforeach; ?>
      </select>
    </div>
  </div>

  <hr>
<?php endif; ?>

<div class="row">
  <div class="col-md-4">
    <label>Montant à encaisser *</label>
    <input type="number" step="0.01" name="amount" class="form-control" required>
  </div>

  <div class="col-md-4">
    <label>Mode de paiement *</label>
    <select name="method" class="form-control" required>
      <option value="">-- Choisir --</option>
      <option value="especes">Espèces</option>
      <option value="cheque">Chèque</option>
      <option value="virement">Virement</option>
      <option value="carte">Carte</option>
    </select>
  </div>

  <div class="col-md-4">
    <label>Référence</label>
    <input type="text" name="reference" class="form-control">
  </div>
</div>

<div class="mt-4">
  <button class="btn btn-success">
    <i class="fas fa-cash-register"></i> Encaisser
  </button>

  <a href="<?= e(base_url('/public/payments/index.php')) ?>" class="btn btn-secondary ml-2">
    Annuler
  </a>
</div>

</form>

</div>
</div>

</div>
</section>

<?php include __DIR__ . '/../../app/views/layout/footer.php'; ?>
