<?php
/******************** DEBUG ********************/
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

/******************** INCLUDES ********************/
require_once __DIR__ . '/../../app/core/helpers.php';
require_once __DIR__ . '/../../app/core/Auth.php';
require_once __DIR__ . '/../../app/config/database.php';
require_once __DIR__ . '/../../app/core/Permissions.php';

Auth::requireLogin();
Permissions::allow(['admin','director','manager','commercial','finance']);

$title  = "Détail Lot | IMMOTECH";
$active = 'lots';

/******************** GET LOT ID ********************/
$lot_id = (int)($_GET['id'] ?? 0);
if ($lot_id <= 0) {
    redirect('/public/lots/index.php');
}

/******************** 1) LOT (infos) ********************/
$stmt = $pdo->prepare("
    SELECT
        l.*,
        p.name  AS project_name,
        lt.name AS lot_type
    FROM lots l
    LEFT JOIN projects p   ON p.id = l.project_id
    LEFT JOIN lot_types lt ON lt.id = l.lot_type_id
    WHERE l.id = ?
    LIMIT 1
");
$stmt->execute([$lot_id]);
$lot = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$lot) {
    redirect('/public/lots/index.php');
}

/******************** 2) VENTE (simulation validée) ********************/
/*
  IMPORTANT :
  - Le prix de vente = simulations.final_price (PAS lots.price)
  - Selon ton historique, le statut peut être 'transformee' ou 'validee'
*/
$stmt = $pdo->prepare("
    SELECT
        s.*,
        c.fullname AS client_name,
        c.phone    AS client_phone
    FROM simulations s
    LEFT JOIN clients c ON c.id = s.client_id
    WHERE s.lot_id = ?
      AND s.status IN ('validee','transformee')
    ORDER BY s.id DESC
    LIMIT 1
");
$stmt->execute([$lot_id]);
$sale = $stmt->fetch(PDO::FETCH_ASSOC); // peut être false

$price_sale = $sale ? (float)$sale['final_price'] : 0.0;

/******************** 3) RÉSERVATIONS (historique) ********************/
$stmt = $pdo->prepare("
    SELECT
        r.*,
        c.fullname AS client_name_db,
        c.phone    AS client_phone_db
    FROM reservations r
    LEFT JOIN clients c ON c.id = r.client_id
    WHERE r.lot_id = ?
    ORDER BY r.id DESC
");
$stmt->execute([$lot_id]);
$reservations = $stmt->fetchAll(PDO::FETCH_ASSOC);

/******************** 4) OVs (payment_orders) ********************/
$stmt = $pdo->prepare("
    SELECT
        po.*,
        s.final_price AS sale_price_ref
    FROM payment_orders po
    LEFT JOIN simulations s ON s.id = po.simulation_id
    WHERE po.lot_id = ?
    ORDER BY po.id DESC
");
$stmt->execute([$lot_id]);
$ovs = $stmt->fetchAll(PDO::FETCH_ASSOC);

/******************** 5) PAIEMENTS (tous paiements du lot via OV) ********************/
$stmt = $pdo->prepare("
    SELECT
        p.*,
        po.id AS ov_id,
        po.amount AS ov_amount,
        po.status AS ov_status
    FROM payments p
    JOIN payment_orders po ON po.id = p.payment_order_id
    WHERE po.lot_id = ?
    ORDER BY p.id DESC
");
$stmt->execute([$lot_id]);
$payments = $stmt->fetchAll(PDO::FETCH_ASSOC);

/******************** 6) SYNTHÈSE FINANCIÈRE (sur prix de vente) ********************/
$stmt = $pdo->prepare("
    SELECT IFNULL(SUM(p.amount),0)
    FROM payments p
    JOIN payment_orders po ON po.id = p.payment_order_id
    WHERE po.lot_id = ?
      AND (p.status = 'valide' OR p.status IS NULL)
");
$stmt->execute([$lot_id]);
$total_paid_lot = (float)$stmt->fetchColumn();

$remaining_sale = ($price_sale > 0) ? max(0, $price_sale - $total_paid_lot) : 0;
$paid_percent_sale = ($price_sale > 0) ? round(($total_paid_lot / $price_sale) * 100, 2) : 0;

$pay_state_label = "Non payé";
$pay_state_badge = "danger";
if ($price_sale <= 0) {
    $pay_state_label = "Aucune vente validée";
    $pay_state_badge = "secondary";
} else {
    if ($total_paid_lot <= 0) {
        $pay_state_label = "Non payé";
        $pay_state_badge = "danger";
    } elseif ($total_paid_lot < $price_sale) {
        $pay_state_label = "Partiellement payé";
        $pay_state_badge = "warning";
    } else {
        $pay_state_label = "Soldé";
        $pay_state_badge = "success";
    }
}

/******************** UI ********************/
include __DIR__ . '/../../app/views/layout/header.php';
include __DIR__ . '/../../app/views/layout/sidebar.php';

function badgeLotStatus(string $status): array {
    return match ($status) {
        'disponible' => ['success', 'Disponible'],
        'reserve'    => ['warning', 'Réservé'],
        'vendu'      => ['primary', 'Vendu'],
        'bloque'     => ['danger', 'Bloqué'],
        default      => ['secondary', $status],
    };
}

function badgeResStatus(string $status): array {
    return match ($status) {
        'en_attente' => ['secondary', 'En attente'],
        'confirmee'  => ['info', 'Confirmée'],
        'vendue'     => ['success', 'Vendue'],
        'livree'     => ['dark', 'Livrée'],
        'expiree'    => ['danger', 'Expirée'],
        default      => ['secondary', $status],
    };
}

function badgeOvStatus(string $status): array {
    return match ($status) {
        'en_attente' => ['warning', 'En attente'],
        'payee'      => ['success', 'Payée'],
        'annulee'    => ['danger', 'Annulée'],
        default      => ['secondary', $status],
    };
}

[$lotBadge, $lotLabel] = badgeLotStatus((string)$lot['status']);
?>
<section class="content pt-3">
<div class="container-fluid">

<div class="d-flex justify-content-between align-items-center mb-3">
  <h3 class="mb-0">
    Lot : <?= e($lot['reference']) ?> — <?= e($lot['lot_type'] ?? '-') ?>
  </h3>
  <a href="<?= e(base_url('/public/lots/index.php')) ?>" class="btn btn-secondary">
    <i class="fas fa-arrow-left"></i> Retour
  </a>
</div>

<!-- 1) INFOS LOT -->
<div class="card mb-3">
  <div class="card-header">
    <h3 class="card-title"><i class="fas fa-building"></i> Informations du lot</h3>
  </div>
  <div class="card-body">
    <div class="row">
      <div class="col-md-6">
        <p><strong>Projet :</strong> <?= e($lot['project_name'] ?? '-') ?></p>
        <p><strong>Type :</strong> <?= e($lot['lot_type'] ?? '-') ?></p>
        <p><strong>N° du lot :</strong> <?= e($lot['reference']) ?></p>
        <p><strong>Côté :</strong>
          <?php
            $side = $lot['side'] ?? '';
            echo $side === 'droite' ? 'À droite' : ($side === 'gauche' ? 'À gauche' : '-');
          ?>
        </p>
      </div>
      <div class="col-md-6">
        <p><strong>Surface :</strong> <?= number_format((float)$lot['surface'], 2, ',', ' ') ?> m²</p>

        <p><strong>Statut lot :</strong>
          <span class="badge badge-<?= e($lotBadge) ?>"><?= e($lotLabel) ?></span>
        </p>

        <p class="mb-0">
          <strong>Prix initial (info) :</strong>
          <?= number_format((float)$lot['price'], 0, ' ', ' ') ?> DA
          <small class="text-muted">(catalogue)</small>
        </p>
      </div>
    </div>
  </div>
</div>

<!-- 2) VENTE (SIMULATION VALIDÉE) -->
<div class="card mb-3">
  <div class="card-header">
    <h3 class="card-title"><i class="fas fa-file-signature"></i> Vente (simulation validée)</h3>
  </div>
  <div class="card-body">
    <?php if (!$sale): ?>
      <div class="alert alert-warning mb-0">
        Aucune simulation validée pour ce lot (prix de vente non défini).
      </div>
    <?php else: ?>
      <div class="row">
        <div class="col-md-6">
          <p><strong>Simulation :</strong> #<?= (int)$sale['id'] ?></p>
          <p><strong>Client :</strong> <?= e($sale['client_name'] ?? '-') ?></p>
          <p class="mb-0"><strong>Téléphone :</strong> <?= e($sale['client_phone'] ?? '-') ?></p>
        </div>
        <div class="col-md-6">
          <p><strong>Prix de vente (référence) :</strong>
            <span class="badge badge-success" style="font-size: 14px;">
              <?= number_format((float)$sale['final_price'], 0, ' ', ' ') ?> DA
            </span>
          </p>
          <p><strong>Remise :</strong> <?= number_format((float)$sale['discount_percent'], 2, ',', ' ') ?> %</p>
          <p class="mb-0"><strong>Date :</strong> <?= e($sale['created_at']) ?></p>
        </div>
      </div>
    <?php endif; ?>
  </div>
</div>

<!-- 6) SYNTHÈSE FINANCIÈRE -->
<div class="card mb-3">
  <div class="card-header">
    <h3 class="card-title"><i class="fas fa-chart-pie"></i> Synthèse financière</h3>
  </div>
  <div class="card-body">
    <div class="row">
      <div class="col-md-3">
        <div class="small text-muted">Prix de vente</div>
        <div style="font-size:18px;"><strong><?= number_format($price_sale, 0, ' ', ' ') ?> DA</strong></div>
      </div>

      <div class="col-md-3">
        <div class="small text-muted">Total encaissé</div>
        <div style="font-size:18px;"><strong><?= number_format($total_paid_lot, 0, ' ', ' ') ?> DA</strong></div>
      </div>

      <div class="col-md-3">
        <div class="small text-muted">Reste à payer</div>
        <div style="font-size:18px;"><strong><?= number_format($remaining_sale, 0, ' ', ' ') ?> DA</strong></div>
      </div>

      <div class="col-md-3">
        <div class="small text-muted">État paiement</div>
        <div>
          <span class="badge badge-<?= e($pay_state_badge) ?>" style="font-size: 14px;">
            <?= e($pay_state_label) ?><?= $price_sale > 0 ? " — ".$paid_percent_sale."%" : "" ?>
          </span>
        </div>
      </div>
    </div>

    <?php if ($price_sale > 0 && $total_paid_lot < $price_sale): ?>
      <div class="alert alert-info mt-3 mb-0">
        Ce lot peut être <strong>Vendu</strong> mais <strong>non soldé</strong>. (Gestion normale des échéanciers / paiements par tranches.)
      </div>
    <?php endif; ?>
  </div>
</div>

<!-- 3) RÉSERVATIONS -->
<div class="card mb-3">
  <div class="card-header">
    <h3 class="card-title"><i class="fas fa-bookmark"></i> Réservations du lot</h3>
  </div>
  <div class="card-body table-responsive p-0">
    <table class="table table-hover mb-0">
      <thead>
        <tr>
          <th>#</th>
          <th>Client</th>
          <th>Réservé le</th>
          <th>Expire le</th>
          <th>Statut</th>
          <th>Action</th>
        </tr>
      </thead>
      <tbody>
      <?php if (!$reservations): ?>
        <tr><td colspan="6" class="text-center text-muted p-4">Aucune réservation.</td></tr>
      <?php else: foreach ($reservations as $r): ?>
        <?php
          [$rb, $rl] = badgeResStatus((string)$r['status']);
          $clientName = $r['client_name_db'] ?? ($r['client_name'] ?? '-');
        ?>
        <tr>
          <td><?= (int)$r['id'] ?></td>
          <td><?= e($clientName) ?></td>
          <td><?= e($r['reserved_at'] ?? '-') ?></td>
          <td><?= e($r['expires_at'] ?? '-') ?></td>
          <td><span class="badge badge-<?= e($rb) ?>"><?= e($rl) ?></span></td>
          <td>
            <a class="btn btn-sm btn-secondary"
               href="<?= e(base_url('/public/reservations/show.php?id='.(int)$r['id'])) ?>">
              <i class="fas fa-eye"></i> Voir
            </a>
          </td>
        </tr>
      <?php endforeach; endif; ?>
      </tbody>
    </table>
  </div>
</div>

<!-- 4) OVs -->
<div class="card mb-3">
  <div class="card-header">
    <h3 class="card-title"><i class="fas fa-file-invoice"></i> Ordres de versement (OV)</h3>
  </div>
  <div class="card-body table-responsive p-0">
    <table class="table table-hover mb-0">
      <thead>
        <tr>
          <th>#</th>
          <th>Simulation</th>
          <th>Montant OV</th>
          <th>Statut</th>
          <th>Date</th>
          <th>Action</th>
        </tr>
      </thead>
      <tbody>
      <?php if (!$ovs): ?>
        <tr><td colspan="6" class="text-center text-muted p-4">Aucun OV.</td></tr>
      <?php else: foreach ($ovs as $po): ?>
        <?php [$ob, $ol] = badgeOvStatus((string)$po['status']); ?>
        <tr>
          <td><?= (int)$po['id'] ?></td>
          <td><?= !empty($po['simulation_id']) ? '#'.(int)$po['simulation_id'] : '-' ?></td>
          <td><?= number_format((float)$po['amount'], 0, ' ', ' ') ?> DA</td>
          <td><span class="badge badge-<?= e($ob) ?>"><?= e($ol) ?></span></td>
          <td><?= e($po['created_at'] ?? '-') ?></td>
          <td>
            <a class="btn btn-sm btn-info"
               href="<?= e(base_url('/public/payment_orders/show.php?id='.(int)$po['id'])) ?>">
              <i class="fas fa-file-invoice"></i> OV
            </a>
          </td>
        </tr>
      <?php endforeach; endif; ?>
      </tbody>
    </table>
  </div>
</div>

<!-- 5) PAIEMENTS -->
<div class="card">
  <div class="card-header">
    <h3 class="card-title"><i class="fas fa-money-bill"></i> Paiements du lot</h3>
  </div>
  <div class="card-body table-responsive p-0">
    <table class="table table-hover mb-0">
      <thead>
        <tr>
          <th>#</th>
          <th>Date</th>
          <th>Montant</th>
          <th>Méthode</th>
          <th>Référence</th>
          <th>OV</th>
          <th>Statut</th>
        </tr>
      </thead>
      <tbody>
      <?php if (!$payments): ?>
        <tr><td colspan="7" class="text-center text-muted p-4">Aucun paiement.</td></tr>
      <?php else: foreach ($payments as $p): ?>
        <tr>
          <td><?= (int)$p['id'] ?></td>
          <td><?= e($p['payment_date'] ?? ($p['created_at'] ?? '-')) ?></td>
          <td><?= number_format((float)$p['amount'], 0, ' ', ' ') ?> DA</td>
          <td><?= e($p['method'] ?? '-') ?></td>
          <td><?= e($p['reference'] ?? '-') ?></td>
          <td>
            <a href="<?= e(base_url('/public/payment_orders/show.php?id='.(int)$p['ov_id'])) ?>">
              #<?= (int)$p['ov_id'] ?>
            </a>
          </td>
          <td><?= e($p['status'] ?? '-') ?></td>
        </tr>
      <?php endforeach; endif; ?>
      </tbody>
    </table>
  </div>
</div>

</div>
</section>

<?php include __DIR__ . '/../../app/views/layout/footer.php'; ?>
